﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;
using Inet.Viewer.Resources;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// A prompt field showing a combobox with default values and optionally another prompt field
    /// of the prompt's type for setting a custom value.
    /// </summary>
    public partial class DefaultValueField : PromptControl
    {
        private static readonly CustomPromptValue customPromptValue = new CustomPromptValue();
        internal PromptControl customPromptField;
        private int promptType;
 
        /// <summary>
        /// Creates the prompt field.
        /// </summary>
        /// <param name="promptData">the prompt data</param>
        /// <param name="customPromptField">the custom prompt field</param>
        /// <param name="range">if it is a range value field</param>
        public DefaultValueField(PromptData promptData, PromptControl customPromptField, bool range)
        {
            InitializeComponent();
            this.customPromptField = customPromptField;
            this.promptType = promptData.Type;
            this.PromptData = promptData;

            ErrorProvider = new ErrorProvider(this);

            customPromptField.Margin = new Padding(0);
            customPromptField.ValueChanged += HandleChanged;

            List<PromptValue> defaultValues = new List<PromptValue>();

            PromptValue alreadySelectedValue = promptData.Values;
            string alreadySelectedValueRepr = alreadySelectedValue == null ? null : alreadySelectedValue.StringRepresentation;
            PromptValue valueToSelect = null;

            bool hasSingleDefaults = false;
            foreach (PromptValue v in promptData.DefaultValues)
            {
                bool rangeValue = v is RangePromptValue;

                if (range == rangeValue) // promptData.RangeAllowed && rangeValue || promptData.DiscreteAllowed && !rangeValue)
                {
                    defaultValues.Add(v);
                    if (v.StringRepresentation == alreadySelectedValueRepr)
                    {
                        valueToSelect = v;
                    }
                }
                if (!rangeValue)
                {
                    hasSingleDefaults = true;
                }
            }
            bool hasDefaults = defaultValues.Count > 0;
            if (promptData.Changeable || range && hasSingleDefaults)
            {
                defaultValues.Insert(0, customPromptValue);
            }
            cmbDefaultValues.DataSource = defaultValues;
            cmbDefaultValues.DisplayMember = "DisplayString";
            cmbDefaultValues.ValueMember = "Value";
            AutoSizeComboBox(cmbDefaultValues);
            customPromptField.Enabled = false;
            if (valueToSelect != null)
            {
                this.cmbDefaultValues.SelectedItem = valueToSelect;
            }
            else if (alreadySelectedValue != null)
            {
                this.cmbDefaultValues.SelectedIndex = 0;
                if (alreadySelectedValue.Value == null)
                {
                    this.chbNoValue.Checked = true;
                }
                else
                {
                    this.cmbDefaultValues.SelectedItem = customPromptValue;
                    customPromptField.Enabled = true;
                }
            }
            else if (!hasDefaults)
            {
                cmbDefaultValues.Enabled = false;
                customPromptField.Enabled = true;
            }
            else if (promptData.Changeable)
            {
                cmbDefaultValues.SelectedIndex = 1;
            }
            if (promptData.Changeable || range && hasSingleDefaults)
            {
                tableLayoutPanel1.Controls.Add(customPromptField, 0, 1 );
                tableLayoutPanel1.SetColumnSpan(customPromptField, 2);
            }
            if (!promptData.Changeable || promptData.MinValue != null)
            {
                chbNoValue.Visible = false;
                chbNoValue.Enabled = false;
            }
            chbNoValue.Text = Inet.Viewer.Resources.strings.Prompt_None_Long;
            ControlForErrors = chbNoValue.Visible ? (Control)chbNoValue : (Control)cmbDefaultValues;
        }

        /// <summary>
        /// Gets the current value.
        /// </summary>
        internal override PromptValue Value
        {
            get 
            {
                if (chbNoValue.Checked)
                {
                    return new SinglePromptValue(null, null, promptType);
                }                    
                PromptValue value = (PromptValue)cmbDefaultValues.SelectedItem;
                if (value == customPromptValue)
                {
                    return customPromptField.Value;
                }
                else
                {
                    return value;
                }
            }
        }

        /// <summary>
        /// Called when the selection was changed of the combobox.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event args</param>
        private void cmbDefaultValues_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cmbDefaultValues.SelectedItem == customPromptValue && !chbNoValue.Checked)
            {
                customPromptField.Enabled = true;
            }
            else
            {
                customPromptField.Enabled = false;
                PromptValue value = Value;
                if (value != null)
                {
                    customPromptField.Value = Value;
                    ValidatePrompt();
                }
            }
        }

        /// <summary>
        /// Called when the no-value checkbox state was changed.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event args</param>
        private void chbNoValue_CheckedChanged(object sender, EventArgs e)
        {
            cmbDefaultValues.Enabled = !chbNoValue.Checked;
            customPromptField.Enabled = cmbDefaultValues.SelectedItem == customPromptValue && !chbNoValue.Checked;
            ValidatePrompt();
        }

        /// <inheritdoc/>
        internal override bool ValidatePrompt()
        {
            return chbNoValue.Checked || customPromptField.ValidatePrompt();
        }

        /// <inheritdoc/>
        public override string ErrorMessage
        {
            get
            {
                return customPromptField.ErrorMessage;
            }
        }
    }
}
